<?php
/* --------------------------------------------------------------
 ConfigurationService.php 2020-03-09
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Configuration;

use Gambio\Core\Configuration\Models\Read;
use Gambio\Core\Configuration\Models\Write;

/**
 * Interface ConfigurationService
 * @package Gambio\Core\Configuration
 */
interface ConfigurationService
{
    /**
     * Searches for a configuration.
     *
     * This method can return null, if no configuration was found with the provided key.
     * Using ::has first ensures to find a valid, configuration.
     *
     * @param string $key Configuration key.
     *
     * @return Read\Interfaces\Configuration|null
     * @see ConfigurationService::has
     */
    public function find(string $key): ?Read\Interfaces\Configuration;
    
    
    /**
     * Searches for a language specific configuration.
     *
     * This method can return null, if no configuration was found with the provided key.
     * Using ::has first ensures to find a valid, language specific configuration (if correct language
     * identifier provided).
     *
     * @param string $key          Configuration key.
     * @param string $languageCode Language code(ISO 639‑1).
     *
     * @return Read\Interfaces\Configuration|null
     * @see ConfigurationService::has
     */
    public function findWithLanguageCode(string $key, string $languageCode): ?Read\Interfaces\Configuration;
    
    
    /**
     * Searches for a language specific configuration.
     *
     * This method can return null, if no configuration was found with the provided key.
     * Using ::has first ensures to find a valid, language specific configuration (if correct language
     * identifier provided).
     *
     * @param string $key
     * @param int    $languageId
     *
     * @return Read\Interfaces\Configuration|null
     */
    public function findWithLanguageId(string $key, int $languageId): ?Read\Interfaces\Configuration;
    
    
    /**
     * Checks if a configuration with the given key exists.
     *
     * @param string $key Configuration key.
     *
     * @return bool
     */
    public function has(string $key): bool;
    
    
    /**
     * Saves the configuration items.
     *
     * This function either adds the configuration, if it not exists or updates
     * an existing configuration. It is possible to provide multiple configurations.
     *
     * @param Write\Interfaces\Configuration ...$configurations
     */
    public function save(Write\Interfaces\Configuration ...$configurations): void;
    
    
    /**
     * Deletes configurations.
     *
     * This function deletes all configuration items with the given keys.
     * If any provided key not exists, the method will silently fail and continue.
     *
     * @param string ...$keys
     */
    public function delete(string ...$keys): void;
}
